# Company:  Delta Elektronika B.V.
# Project:  PSC-ANY-EXT_Examples
# File:     CANopen.py
# Date:     02/06/2025

# This code is provided "as is" without any guarantees or warranties. Delta Elektronika B.V.
# is not responsible for any damages, losses, or issues arising from its use, implementation, or modification.

import canopen
import os
import time

"""
Class: CANopen

Description:
    A class within PSC-ANY-EXT that interacts with and controls 
    a Delta Elektronika power supply using the CANopen protocol. 
    Operation has been testing with the "PCAN-USB": https://www.peak-system.com/PCAN-USB.199.0.html?L=1
"""

class CANopen:
    def __init__(self, address, dataformat, own_address, speed):
        if own_address == None:                                                                                                                                                                             # Error catching
            raise ValueError(f"CANopen requires an extra argument passed with the PSC_ANY_EXT called 'own_address'.")                                                                                       # ..
        if speed not in {'50000', '125000', '250000', '500000', '800000', '1000000'}:                                                                                                     # ..
            raise ValueError(f"CANopen requires an extra argument passed with the PSC_ANY_EXT called 'speed' with options ['50000', '125000', '250000', '500000', '800000', '1000000'] ") # ..
        self.dataformat = dataformat                                                                    # Local variable declaration.
        self.speed = speed                                                                              # ..
        self.current_dir = os.path.dirname(os.path.abspath(__file__))                                   # Get PSC-ANY-EXT.eds path.
        self.eds_file_path = os.path.join(self.current_dir, 'PSC-ANY-EXT.eds')                          # ..
        self.client = canopen.Network()                                                                 # Define the CANopen client.
        self.PSC_ANY_EXT_Node = canopen.RemoteNode(int(address), self.eds_file_path)                    # ..
        self.client.add_node(self.PSC_ANY_EXT_Node)                                                     # add PSC-ANY-EXT to the network.


    def connect(self):                                                                                  # Connect to the Modbus/TCP bus.
        self.client.connect(bustype='pcan', channel='PCAN_USBBUS1', bitrate=self.speed)                 # ..
        self.client.scanner.search()                                                                    # Make sure PSC-ANY-EXT is connected.
        time.sleep(0.05)                                                                                # ..
        if self.PSC_ANY_EXT_Node.id not in self.client.scanner.nodes:                                   # ..
            raise ValueError(f"The PSC-ANY-EXT node ({self.PSC_ANY_EXT_Node.id}) is not connected. Is it connected and/or set-up correctly?")
        self.PSC_ANY_EXT_Node.nmt.state = 'PRE-OPERATIONAL'                                             # CANopen State: PRE-OPERATIONAL


    def request(self, parameter):                                                                       # Request a parameter value from the Delta Elektronika powersupply using CANopen:

        if self.dataformat == 'float_format_a':                                                         # Using Float Format.
            match parameter.lower():                                                                    # ..
                case 'cvprg':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200C][1].raw                                     # ..
                case 'cvmon':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200C][2].raw                                     # ..
                case 'ccprg':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200C][3].raw                                     # ..
                case 'ccmon':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200C][4].raw                                     # ..
                case 'cvlim':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200C][5].raw                                     # ..
                case 'cclim':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200C][6].raw                                     # ..
                case 'refresh':                                                                         # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200D][0].raw                                     # ..
                case 'status_registera':                                                                # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200E][0].raw                                     # ..
                case 'status_registerb':                                                                # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x200F][0].raw                                     # ..
                case _:
                    raise ValueError(f"Incorrect parameter ({parameter}) has been requested")           # Error

        if self.dataformat == '16bit_format_a':                                                         # Using 16-Bit Format.
            match parameter.lower():                                                                    # ..
                case 'cvprg':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x2016][1].raw                                     # ..
                case 'cvmon':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x2016][2].raw                                     # ..
                case 'ccprg':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x2016][3].raw                                     # ..
                case 'ccmon':                                                                           # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x2016][4].raw                                     # ..
                case 'refresh':                                                                         # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x2017][0].raw                                     # ..
                case 'status_registera':                                                                # ..
                    return self.PSC_ANY_EXT_Node.sdo[0x2018][0].raw                                     # ..
                case _:
                    raise ValueError(f"Incorrect parameter ({parameter}) has been requested")           # Error


    def set(self, parameter, value):                                                                    # Set a parameter value on the Delta Elektronika powersupply using CANopen:

        if self.dataformat == 'float_format_a':                                                         # Using Float Format.
            match parameter.lower():                                                                    # ..
                case 'cvprg':                                                                           # ..
                    self.PSC_ANY_EXT_Node.sdo[0x200A][1].raw = value                                    # ..
                case 'ccprg':                                                                           # ..
                    self.PSC_ANY_EXT_Node.sdo[0x200A][2].raw = value                                    # ..
                case 'remctrl':                                                                         # ..
                    value = self.int_to_struct(value)                                                   # Convert UINT16 to byte structure.
                    self.PSC_ANY_EXT_Node.sdo[0x200B][0].raw = value                                    # ..
                case _:
                    raise ValueError(f"Incorrect parameter ({parameter}) passed as an argument.")       # Error

        if self.dataformat == '16bit_format_a':                                                         # Using 16-Bit Format.
                match parameter.lower():                                                                # ..
                    case 'cvprg':                                                                       # ..
                        self.PSC_ANY_EXT_Node.sdo[0x2014][1].raw = value                                # ..
                    case 'ccprg':                                                                       # ..
                        self.PSC_ANY_EXT_Node.sdo[0x2014][2].raw = value                                # ..
                    case 'remctrl':                                                                     # ..
                        value = self.int_to_struct(value)                                               # Convert UINT16 to byte structure.
                        self.PSC_ANY_EXT_Node.sdo[0x2015][0].raw = value                                # ..
                    case _:                                                                             # ..
                        raise ValueError(f"Incorrect parameter ({parameter}) passed as an argument.")   # Error


    def disconnect(self):                                                                               # Disconnect from the CANopen bus.
        self.client.disconnect()


    def int_to_struct(self, value):                                                                         # Pack integer to struct.
        packed = value.to_bytes(2, byteorder='little')                                                      # ..
        return packed                                                                                       # ..