# Company:  Delta Elektronika B.V.
# Project:  PSC-ANY-EXT_Examples
# File:     ModbusTCP.py
# Date:     02/06/2025

# This code is provided "as is" without any guarantees or warranties. Delta Elektronika B.V.
# is not responsible for any damages, losses, or issues arising from its use, implementation, or modification.

from pyModbusTCP.client import ModbusClient
import struct

"""
Class: ModbusTCP

Description:
    A class within PSC-ANY-EXT that interacts with and controls 
    a Delta Elektronika power supply using the Modbus protocol. 
    Communication speed of this class is set-up automatically according to fieldbus speed set on the PSC-ANY-EXT.
"""

class ModbusTCP:
    def __init__(self, address, dataformat):
        self.address = address                                                                  # Save entered address in variable.
        self.dataformat = dataformat                                                            # Save entered dataformat.
        self.client = None                                                                      # define the ModbusTCP client.

    def connect(self):                                                                          # Connect to the Modbus/TCP bus.
        self.client = ModbusClient(host=self.address, port=502, unit_id=1, auto_open=True)      # ..

    def request(self, parameter):                                                               # Request a parameter value from the Delta Elektronika powersupply using ModbusTCP.

        if self.dataformat == 'float_format_a':                                                 # Using Float Format:
            match parameter.lower():                                                            # ..
                case 'cvprg':                                                                   # ..
                    cvprg_regs = self.client.read_holding_registers(0x800, 2)                   # ..
                    cvprg = self.parse_to_float(cvprg_regs[1], cvprg_regs[0])                   # ..
                    return cvprg                                                                # ..
                case 'cvmon':                                                                   # ..
                    cvmon_regs = self.client.read_holding_registers(0x802, 2)                   # ..
                    cvmon = self.parse_to_float(cvmon_regs[1], cvmon_regs[0])                   # ..
                    return cvmon                                                                # ..
                case 'ccprg':                                                                   # ..
                    ccprg_regs = self.client.read_holding_registers(0x804, 2)                   # ..
                    ccprg = self.parse_to_float(ccprg_regs[1], ccprg_regs[0])                   # ..
                    return ccprg                                                                # ..
                case 'ccmon':                                                                   # ..
                    ccmon_regs = self.client.read_holding_registers(0x806, 2)                   # ..
                    ccmon = self.parse_to_float(ccmon_regs[1], ccmon_regs[0])                   # ..
                    return ccmon                                                                # ..
                case 'cvlim':                                                                   # ..
                    cvmax_regs = self.client.read_holding_registers(0x808, 2)                   # ..
                    cvmax = self.parse_to_float(cvmax_regs[1], cvmax_regs[0])                   # ..
                    return cvmax                                                                # ..
                case 'cclim':                                                                   # ..
                    ccmax_regs = self.client.read_holding_registers(0x80A, 2)                   # ..
                    ccmax = self.parse_to_float(ccmax_regs[1], ccmax_regs[0])                   # ..
                    return ccmax                                                                # ..
                case 'refresh':                                                                 # ..
                    update_counter = self.client.read_holding_registers(0x80C, 1)               # ..
                    return update_counter[0]                                                    # ..
                case 'status_registera':                                                        # ..
                    status_registera = self.client.read_holding_registers(0x80D, 1)             # ..
                    return status_registera[0]                                                  # ..
                case 'status_registerb':                                                        # ..
                    status_registerb = self.client.read_holding_registers(0x80E, 1)             # ..
                    return status_registerb[0]                                                  # ..
                case _:
                    raise ValueError(f"Incorrect parameter has been requested: {parameter}")    # Error

        elif self.dataformat == '16bit_format_a':                                               # 16-Bit Format:
            match parameter.lower():                                                            # ..
                case 'cvprg':                                                                   # ..
                    cvprg = self.client.read_holding_registers(0x800, 1)                        # ..
                    return cvprg[0]                                                             # ..
                case 'cvmon':                                                                   # ..
                    cvmon = self.client.read_holding_registers(0x801, 1)                        # ..
                    return cvmon[0]                                                             # ..
                case 'ccprg':                                                                   # ..
                    ccprg = self.client.read_holding_registers(0x802, 1)                        # ..
                    return ccprg[0]                                                             # ..
                case 'ccmon':                                                                   # ..
                    ccmon = self.client.read_holding_registers(0x803, 1)                        # ..
                    return ccmon[0]                                                             # ..
                case 'refresh':                                                                 # ..
                    update_counter = self.client.read_holding_registers(0x804, 1)               # ..
                    return update_counter[0]                                                    # ..
                case 'status_registera':                                                        # ..
                    status_registera = self.client.read_holding_registers(0x805, 1)             # ..
                    return status_registera[0]                                                  # ..
                case _:
                    raise ValueError(f"Incorrect parameter has been requested: {parameter}")    # Error

    def set(self, parameter, value):                                                            # Set a parameter value of the Delta Elektronika powersupply using ModbusTCP:
        if self.dataformat == 'float_format_a':                                                 # Using Float Format:
            match parameter.lower():
                case 'cvprg':
                    low, high = self.float_to_16bit_registers(value)                            # Set CVprogram
                    self.client.write_multiple_registers(0x0, [high, low])                      # ..
                case 'ccprg':
                    low, high = self.float_to_16bit_registers(value)                            # Set CCprogram
                    self.client.write_multiple_registers(0x2, [high, low])                      # ..
                case 'remctrl':
                    value = int(value)                                                          # Set RemCTRL
                    self.client.write_single_register(0x4, value)                               # ..
                case _:
                    raise ValueError(f"Incorrect parameter has been requested: {parameter}")

        elif self.dataformat == '16bit_format_a':                                               # 16-Bit Format:
            match parameter.lower():
                case 'cvprg':
                    value = int(value)                                                          # Set CVprogram
                    self.client.write_single_register(0x0, value)                               # ..
                case 'ccprg':
                    value = int(value)                                                          # Set CCprogram
                    self.client.write_single_register(0x1, value)                               # ..
                case 'remctrl':
                    value = int(value)                                                          # Set RemCTRL
                    self.client.write_single_register(0x2, value)                               # ..
                case _:
                    raise ValueError(f"Incorrect parameter has been requested: {parameter}")    # Error

    def disconnect(self):                                                                       # Disconnect from the Modbus/TCP bus.
        self.client.close()                                                                     # ..


    def parse_to_float(self, high, low):                                                   # parse 2, 16-bit values to a float.
        combined = (high << 16) | low                                                      # ..
        float_value = struct.unpack('>f', struct.pack('>I', combined))[0]   # ..
        return float_value                                                                 # ..

    def float_to_16bit_registers(self, value):                                             # parse float to 2 integers.
        packed = struct.pack('>f', value)                                          # Pack float to bytes.
        high, low = struct.unpack('>HH', packed)                                     # unpack bytes to words.
        return high, low                                                                   # ..