# Company:  Delta Elektronika B.V.
# Project:  PSC_ANY_EXT_Examples
# File:     PSC_ANY_EXT.py
# Date:     02/06/2025

# This code is provided "as is" without any guarantees or warranties. Delta Elektronika B.V.
# is not responsible for any damages, losses, or issues arising from its use, implementation, or modification.

from Fieldbuses.ModbusTCP.ModbusTCP import ModbusTCP
from Fieldbuses.EthernetIP.EthernetIP import EthernetIP
from Fieldbuses.CANopen.CANopen import CANopen
from colorama import Fore, Style

"""
Class: PSC_ANY_EXT

Description:
    A class to interact with and control a Delta Elektronika power supply using 
    different fieldbus protocols. It provides methods 
    to connect, request parameters, and disconnect from the power supply.

Attributes:
    AVAILABLE_FIELDBUSES (dict): A dictionary mapping fieldbus types to their respective classes.

Methods:
    connect(): Establishes a connection to the selected fieldbus.
               request(parameter): Requests a specified parameter from the power supply.
    disconnect(): Disconnects from the fieldbus.

Parameters: 
    the available built-in parameters are:  CVprg, CCprg, POWprg, CVmon, CCmon, POWmon, CVlim, CClim, POWmax, RemCTRL, 
                                            Counter, Status_RegisterA, Status_RegisterB.
"""

class PSC_ANY_EXT:

    def __init__(self, fieldbus, address, dataformat='None', own_address=None, speed=None):
        self.correctoperation = True                                                                                    # Initialise correctoperation as correct.
        self.fieldbustype = fieldbus                                                                                    # Save entered fieldbus type in variable.
        self.dataformat = dataformat;                                                                                   # Save dataformat so that it can be used in main.py
        if dataformat.lower() not in {'float_format_a', '16bit_format_a'}:                                                  # ..
            self.printerror(f"Unsupported dataformat: {dataformat}")                                                    # ..
        try:
            if fieldbus == 'modbus_tcp':                                                                                # Dynamically choose the correct fieldbus class based on the fieldbus type
                self.fieldbus = ModbusTCP(address, dataformat.lower())                                                  # ..
            elif fieldbus == 'ethernet_ip':                                                                             # ..
                self.fieldbus = EthernetIP(address, dataformat.lower())                                                 # ..
            elif fieldbus == 'canopen':                                                                                 # ..
                self.fieldbus = CANopen(address, dataformat.lower(), own_address, speed)                                # ..
            else:                                                                                                       # ..
                self.printerror(f"Unsupported fieldbus type: {fieldbus}")                                               # ..
        except Exception as e:                                                                                          # ..
            self.printerror(f"Failed to initialise {self.fieldbustype} bus: {e}")                                       # ..

    def connect(self):                                                                                                  # Connect to the chosen fieldbus.
        if self.correctoperation == True:                                                                               # ..
            if self.fieldbustype.lower() in {'modbus_tcp' : ModbusTCP, 'ethernet_ip' : EthernetIP, 'canopen' : CANopen}:# ..
                try:                                                                                                    # ..
                    self.fieldbus.connect()                                                                             # ..
                except Exception as e:                                                                                  # ..
                    self.printerror(f"Failed to connect to {self.fieldbustype} bus: {e}")                               # ..
                    return -1                                                                                           # ..
                print(Fore.GREEN + f'{self.fieldbustype} set-up.')                                                      # ..
                print(Style.RESET_ALL)                                                                                  # ..
            else:                                                                                                       # ..
                self.printerror(f"Failed to connect to {self.fieldbustype} bus: unsupported fieldbus.")                 # ..
                return -1                                                                                               # ..
        else:                                                                                                           # ..
            return -1                                                                                                   # ..

    def request(self, parameter):                                                                                       # Request a parameter value from the Delta Elektronika powersupply using the chosen communication protocol.
        if self.correctoperation == True:                                                                               # ..
            try:                                                                                                        # ..
                return self.fieldbus.request(parameter)                                                                 # ..
            except Exception as e:                                                                                      # ..
                self.printerror(f"Failed to acquire {parameter}: {e}")                                                  # ..

    def set(self, parameter, value):                                                                                    # Set a parameter value on the Delta Elektronika powersupply using the chosen communication protocol.
        if self.correctoperation == True:                                                                               # ..
            try:                                                                                                        # ..
                return self.fieldbus.set(parameter, value)                                                              # ..
            except Exception as e:                                                                                      # ..
                self.printerror(f"Failed to set {parameter} to {value}: {e}")                                           # ..
                return -1                                                                                               # ..

    def disconnect(self):                                                                                               # Disconnect from the bus.
        try:                                                                                                            # ..
            self.fieldbus.disconnect()                                                                                  # ..
        except Exception as e:                                                                                          # ..
            self.printerror(f"Failed to disconnect from {self.fieldbustype} bus: {e}")                                  # ..
        print(Fore.GREEN + f'Disconnected from {self.fieldbustype}.')                                                   # ..
        print(Style.RESET_ALL)                                                                                          # ..


    def printerror(self, Message):                                                                                      # Print a magenta error in the terminal.
        self.correctoperation = False                                                                                   # ..
        print(Fore.MAGENTA + f"ERROR: {Message}")                                                                       # ..
        print(Style.RESET_ALL)                                                                                          # ..