# Company:          Delta Elektronika B.V.
# Author:           Robbert Berrevoets
# Project:          PSC-ANY-EXT-Examples
# Compatibility:    PSC-ANY-EXT
# File:             main.py
# Date:             02/06/2025

# This code is provided "as is" without any guarantees or warranties. Delta Elektronika B.V.
# is not responsible for any damages, losses, or issues arising from its use, implementation, or modification.

from PSC_ANY_EXT import PSC_ANY_EXT
import time

"""
Description:
    This project demonstrates how to interact with and control a Delta Elektronika power supply
    using the PSC-ANY-EXT. The main script connects to the power supply using the chosen fieldbus, 
    requests parameters such as voltage, current, and various status indicators, 
    and prints these parameters in a readable format. 

    INT-MOD-ANY Class Parameters:
        fieldbus                    : 'ethernet_ip', 'modbus_tcp', 'canopen'
        address                     : IP-address or Node-address (e.g. '10.1.0.137' or '5' respectively)
        dataformat                  : 'float_format_a', '16bit_format_a'
        own_address (CANopen only)  : Own node-address (e.g. '3' or '29')
        speed       (CANopen only)  : '50000', '125000', '250000', '500000', '800000', '1000000' (10000 and 20000 not supported by this project)
       
    Settable Parameters             : 'remctrl', 'cvprg', 'ccprg'
        
    Requestable Parameters:    
        Using 'float_format_a'      : cvprg, ccprg, cvmon, ccmon, cvlim, cclim, status_registera, status_registerb and refresh.
        Using '16bit_format_a'      : cvprg, ccprg, cvmon, ccmon, status_registera and refresh.

Instructions:
1. Follow the manual to set-up the PSC-ANY-EXT.
2. install the following python packages: canopen~=2.3.0, pyModbusTCP~=0.3.0, pycomm3~=1.2.14, colorama~=0.4.6
3. Edit the instance of the PSC-ANY-EXT class in this file, so that it is compatible with the fieldbus used.
4. Set the powersupply.set() variables to the desired values.
"""

def main():
    # Connecting with the power supply.
    powersupply = PSC_ANY_EXT(
        fieldbus='canopen',
        address='1',
        dataformat= 'float_format_a',
        own_address='2',
        speed='500000'
    )

    # Could not make connection.
    if powersupply.connect() == -1:
        powersupply.disconnect()
        while True:
            pass

    # Requesting and printing parameters: Float Format
    if powersupply.dataformat == 'float_format_a':

        # Set CVprg and CCprg
        powersupply.set('cvprg', 0)     # [0 .. CVlim]
        powersupply.set('ccprg', 0)     # [0 .. CClim]

        # Set RemCTRL
        powersupply.set('remctrl', 0)   # [0 .. 65535]

        time.sleep(0.2)                                # Give Power Supply time to settle. Adjust accordingly.

        print("\n")
        print("====================================")
        print(f"CVlim:                  {powersupply.request('cvlim'):.3f}V")
        print(f"CClim:                  {powersupply.request('cclim'):.3f}A")
        print()
        print(f"CVprg:                  {powersupply.request('cvprg'):.3f}V")
        print(f"CCprg:                  {powersupply.request('ccprg'):.3f}A")
        print()
        print(f"CVmon:                  {powersupply.request('cvmon'):.3f}V")
        print(f"CCmon:                  {powersupply.request('ccmon'):.3f}A")
        print()
        print(f"Status_RegisterA:       {powersupply.request('status_registera')}")
        print(f"Status_RegisterB:       {powersupply.request('status_registerb')}")
        print()
        print(f"Refresh-Counter:        {powersupply.request('refresh')}")
        print("====================================")
        print("\n")

    # Requesting and printing parameters: 16Bit-Format
    elif powersupply.dataformat == '16bit_format_a':

        # Set CVprg and CCprg
        powersupply.set('cvprg', 0)  # [0 .. 65535]
        powersupply.set('ccprg', 0)  # [0 .. 32768]

        # Set RemCTRL
        powersupply.set('remctrl', 0)   # [0 .. 65535]

        time.sleep(0.1)                                # Give Power Supply time to settle. Adjust accordingly.

        print("\n")
        print("====================================")
        print(f"CVprg:                  {powersupply.request('cvprg')} Bits")
        print(f"CCprg:                  {powersupply.request('ccprg')} Bits")
        print()
        print(f"CVmon:                  {powersupply.request('cvmon')} Bits")
        print(f"CCmon:                  {powersupply.request('ccmon')} Bits")
        print()
        print(f"Status_RegisterA:       {powersupply.request('status_registera')}")
        print()
        print(f"Refresh-Counter:        {powersupply.request('refresh')}")
        print("====================================")
        print("\n")

    # Disconnecting from the power supply.
    powersupply.disconnect()


if __name__ == "__main__":
    main()
